/**
 * Encodes a public key.
 * @param unencodedKey - unencoded public key
 * @returns encoded public key
 */
export declare function getPublicKeyEncoded(unencodedKey: string): string;
/**
 * Unencodes a public key.
 * @param publicKey - Encoded public key
 * @returns decoded public key
 */
export declare function getPublicKeyUnencoded(publicKey: string): string;
/**
 * Converts a WIF to private key.
 */
export declare function getPrivateKeyFromWIF(wif: string): string;
/**
 * Converts a private key to WIF.
 */
export declare function getWIFFromPrivateKey(privateKey: string): string;
/**
 * Converts a private key to public key.
 * @param privateKey - 64 bit private key
 * @param encode - returns the encoded form if true.
 */
export declare function getPublicKeyFromPrivateKey(privateKey: string, encode?: boolean): string;
/**
 * Converts a public key to verification script form.
 * VerificationScript serves a very niche purpose.
 * It is attached as part of the signature when signing a transaction.
 * Thus, the name 'scriptHash' instead of 'keyHash' is because we are hashing the verificationScript and not the PublicKey.
 */
export declare function getVerificationScriptFromPublicKey(publicKey: string): string;
/**
 * Extracts the public key from the verification script. This only works for single key accounts.
 * @param script - hexstring
 */
export declare function getPublicKeyFromVerificationScript(script: string): string;
/**
 * Converts a verification script to scripthash.
 *
 * @param verificationScript - hexstring
 */
export declare function getScriptHashFromVerificationScript(verificationScript: string): string;
/**
 * Converts a public key to scripthash.
 */
export declare function getScriptHashFromPublicKey(publicKey: string): string;
/**
 * Converts a scripthash to address.
 */
export declare function getAddressFromScriptHash(scriptHash: string, addressVersion?: number): string;
/**
 * Converts an address to scripthash.
 */
export declare function getScriptHashFromAddress(address: string): string;
/**
 * Generates a random private key.
 */
export declare function generatePrivateKey(): string;
/**
 * Extracts the address version from a given correct NEO address.
 * @param address - A NEO address string.
 * @returns Address version
 */
export declare function getAddressVersion(address: string): number;
//# sourceMappingURL=core.d.ts.map