import { Account, AccountJSON } from "./Account";
import { ScryptParams } from "./nep2";
export interface WalletJSON {
    name: string;
    version: string;
    scrypt: ScryptParams;
    accounts: AccountJSON[];
}
/**
 * File to store private keys according to the NEP-2 specification.
 */
export declare class Wallet {
    name: string;
    version: string;
    scrypt: ScryptParams;
    accounts: Account[];
    constructor(obj?: Partial<WalletJSON>);
    get [Symbol.toStringTag](): string;
    /**
     * Returns the default Account according to the following rules:
     * 1. First Account where isDefault is true.
     * 2. First Account with a decrypted private key.
     * 3. First Account with an encrypted private key.
     * 4. First Account in the array.
     * Throws error if no accounts available.
     */
    get defaultAccount(): Account;
    /**
     * Adds an account.
     * @param acct - Account or WalletAccount object.
     * @returns index position of Account in array.
     */
    addAccount(acct: Account | AccountJSON): number;
    /**
     * Attempts to decrypt Account at index in array.
     * @param index - index of Account in array.
     * @param keyphrase - keyphrase
     * @returns Promise of decryption success/failure.
     */
    decrypt(index: number, keyphrase: string): Promise<boolean>;
    /**
     * Attempts to decrypt all accounts with keyphrase.
     * @returns Promise of accounts decryption success/failure .
     */
    decryptAll(keyphrase: string): Promise<boolean[]>;
    /**
     * Attempts to encrypt Account at index in array.
     * @param index - index of Account in array.
     * @param keyphrase - keyphrase
     * @returns Promise of encryption success/failure.
     */
    encrypt(index: number, keyphrase: string): Promise<boolean>;
    /**
     * Attempts to encrypt all accounts with keyphrase.
     * @param keyphrase - keyphrase
     * @returns Promise of accounts encryption success/failure.
     */
    encryptAll(keyphrase: string): Promise<boolean[]>;
    /**
     * Export this class as a JS object.
     */
    export(): WalletJSON;
    /**
     * Set Account at index in array to be default account.
     * @param index - index of the Account in accounts array.
     */
    setDefault(index: number): this;
}
export default Wallet;
//# sourceMappingURL=Wallet.d.ts.map