export interface EcdsaSignature {
    r: string;
    s: string;
}
/**
 * Interface for providing elliptic cryptography methods. Instead of instantiating your own curve, you should be getting a curve through getCurve.
 *
 * @example
 *
 * const curve = getCurve(EllipticCurvePreset.SECP256R1);
 * const message = tx.serialize();
 * const signature = curve.sign(message, privateKey);
 */
export declare class EllipticCurve {
    private curve;
    constructor(preset: string);
    /**
     * Signs a message with the given private key.
     * @param message - hexstring message.
     * @param privateKey - hexstring.
     * @param k - number or hexstring between 0 and the n parameter of the curve. Provide this if you wish to generate a deterministic signature. Do note that there are security implications if you do.
     */
    sign(message: string, privateKey: string, k?: number | string): EcdsaSignature;
    /**
     * Validates a signature against the original message and the public key of the signing key.
     * @param message - the original hexstring message.
     * @param signature - the signature.
     * @param publicKey - encoded or unencoded public key.
     */
    verify(message: string, signature: EcdsaSignature, publicKey: string): boolean;
    /**
     * Generates the public key from the given private key. Encoding it results in a shorter public key.
     * @param privateKey - 64 byte hexstring.
     * @param encode - whether to return the compressed form.
     */
    getPublicKey(privateKey: string, encode?: boolean): string;
    /**
     * Decodes an encoded public key to its unencoded form.
     * @param publicKey - 33 byte hexstring starting with 02 or 03.
     */
    decodePublicKey(publicKey: string): string;
}
export declare enum EllipticCurvePreset {
    SECP256R1 = 0,
    SECP256K1 = 1
}
export declare function getCurve(curveName: EllipticCurvePreset): EllipticCurve;
//# sourceMappingURL=curve.d.ts.map