import BN from "bignumber.js";
/**
 * A fixed point notation used widely in the NEO system for representing decimals.
 * It uses a signed 64-bit integer divided by 10^8. This allows it to accurately represent 8 decimal places.
 * Supports up to 8 decimals and is 8 bytes long.
 *
 * @deprecated Please use BigInteger instead. This will be removed in the first minor update.
 * @example
 * const zero = new Fixed8();
 * const fromNumber = new Fixed8(12345);
 * const fromString = new Fixed8("12345");
 * const fromHex = Fixed8.fromHex("3039");
 * const fromReverseHex = Fixed8.fromReverseHex("3930");
 *
 */
export declare class Fixed8 extends BN {
    /**
     * Maximum value that a Fixed8 can hold (92233720368.54775807).
     */
    static readonly MAX_VALUE: Fixed8;
    /**
     * Minimum value that a Fixed8 can hold (-9223372036854775808).
     */
    static readonly MIN_VALUE: Fixed8;
    static fromHex(hex: string): Fixed8;
    static fromReverseHex(hex: string): Fixed8;
    /**
     * Creates a new Fixed8 from a number that is too large to fit in a Fixed8.
     *
     * @remarks
     * The constructor of Fixed8 only accepts input between the MAX_VALUE and MIN_VALUE.
     * However, some inputs from RPC are larger than that due to them sending the raw value over.
     * This method allows the creation of the Fixed8 by stating the number of decimals to shift.
     * If after shifting the number is still out of range, this method will throw.
     *
     * @param input - a string or number that represents a number too big for a Fixed8.
     * @param decimals - the number of decimals to shift by. We will simply divide the number by 10^decimals. Defaults to 8.
     *
     * @example
     * const rawValue = "922337203680"; // RPC may send this value representing 9223.37203680 GAS.
     *
     * const safe = Fixed8.fromRawNumber(rawValue, 8);
     * console.log(safe.toString()); // 9223.37203680
     *
     * const willThrow = new Fixed8(rawValue);
     */
    static fromRawNumber(input: string | number, decimals?: number): Fixed8;
    constructor(input?: number | string | BN, base?: number);
    toHex(): string;
    toReverseHex(): string;
    /**
     * Returns a raw number represetation of Fixed8.
     */
    toRawNumber(): BN;
    /**
     * Returns a Fixed8 whose value is rounded upwards to the next whole number.
     */
    ceil(): Fixed8;
    /**
     * Returns a Fixed8 whose value is rounded downwards to the previous whole number.
     */
    floor(): Fixed8;
    /**
     * Returns true if the value is equivalent.
     */
    equals(other: string | number | Fixed8 | BN): boolean;
    /**
     * Returns a Fixed8 rounded to the nearest dp decimal places according to rounding mode rm.
     * If dp is null, round to whole number.
     * If rm is null, round according to default rounding mode.
     * @param dp - number of decimal places to keep. Defaults to 0.
     * @param rm - rounding mode.
     */
    round(dp?: number, rm?: BN.RoundingMode): Fixed8;
    /**
     * Returns a Fixed8 whose value is the value of this Fixed8 divided by `n`
     */
    dividedBy(n: string | number | BN, base?: number): Fixed8;
    /**
     * {@inheritDoc Fixed8.dividedBy}
     */
    div(n: string | number | BN, base?: number): Fixed8;
    /**
     * Returns a Fixed8 whose value is the value of this Fixed8 multipled by `n`
     */
    times(n: string | number | BN, base?: number): Fixed8;
    /**
     * {@inheritDoc Fixed8.times}
     */
    mul(n: string | number | BN, base?: number): Fixed8;
    /**
     * Returns a Fixed8 whose value is the value of this Fixed8 plus `n`
     */
    plus(n: string | number | BN, base?: number): Fixed8;
    /**
     * {@inheritDoc Fixed8.plus}
     */
    add(n: string | number | BN, base?: number): Fixed8;
    /**
     * Returns a Fixed8 whose value is the value of this Fixed8 minus `n`
     */
    minus(n: string | number | BN, base?: number): Fixed8;
    /**
     * {@inheritDoc Fixed8.minus}
     */
    sub(n: string | number | BN, base?: number): Fixed8;
}
export default Fixed8;
//# sourceMappingURL=Fixed8.d.ts.map