import { HexString } from "./HexString";
/**
 * This is a BigInteger representation, providing several helper methods to convert between different formats as well as basic math operations.
 * Mainly used within the NEO VM for representing numbers.
 */
export declare class BigInteger {
    #private;
    /**
     * Creates a BigInteger from a twos complement hexstring.
     * @param hexstring - HexString class or a string.
     * @param littleEndian - indicate if provided string is big or little endian. Defaults to false(big endian).
     *
     * @example
     * const negativeOne = BigInteger.fromTwos("ff");
     * const oneTwoEight = BigInteger.fromTwos("0080");
     *
     * // Apply a second parameter when using little endian.
     * const oneTwoEightAgain = BigInteger.fromTwos("8000", true);
     */
    static fromTwos(hexstring: HexString): BigInteger;
    static fromTwos(hexstring: string, littleEndian?: boolean): BigInteger;
    /**
     * Creates a BigInteger from a hexstring.
     * @param hexstring - HexString class or a bigendian string.
     *
     * @example
     * const twoFiveFive = BigInteger.fromHex("ff");
     * const oneTwoEight = BigIntger.fromHex("80");
     * const oneTwoEightAgain = BigInteger.fromHex("0080");
     *
     * // Apply a second parameter when using little endian.
     * const oneSixThreeEightFour = BigInteger.fromHex("0040", true);
     */
    static fromHex(hexstring: HexString): BigInteger;
    static fromHex(hexstring: string, littleEndian?: boolean): BigInteger;
    /**
     * Creates a BigInteger from a number.
     * @param input - Javascript number or string containing numbers.
     *
     * @throws non-integer provided
     * @throws Input is not a stringified number.
     *
     * @example
     * const zero = BigInteger.fromNumber(0);
     * const negativeOne = BigInteger.fromNumber(-1);
     *
     * const usingString = BigInteger.fromNumber("-1");
     */
    static fromNumber(input: number | string): BigInteger;
    /**
     * Creates a BigInteger instance from a decimal by parsing the decimals and shifting the decimal point by a provided number of places.
     *
     * This is mainly used with dealing with Nep17 tokens.
     * While most tokens support some sort of decimal places, the data is actually stored as an integer.
     * This helper method converts the decimal number to the integer representation to work with.
     * To convert back, use toDecimal(decimals);
     * @param input - Javascript number or string containing numbers. Accepts decimals.
     * @param decimals - Number of decimal places to support.
     *
     * @example
     *
     * const transformedDecimal = BigInteger.fromDecimal(12.34,3);
     * console.log(transformedDecimal.toString()); // 12340
     *
     * const oneGas = BigInteger.fromDecimal(1,8);
     * console.log(oneGas); // 100000000
     */
    static fromDecimal(input: number | string, decimals: number): BigInteger;
    private constructor();
    /**
     * Returns a big endian hex representation of the integer.
     * Does not come with a prefix.
     */
    toHex(): string;
    /**
     * Returns a little endian hex representation of the integer.
     * Does not come with a prefix.
     */
    toReverseHex(): string;
    /**
     * Returns a big endian two's complement representation of the integer.
     * Does not come with a prefix.
     */
    toTwos(): string;
    /**
     * Returns a little endian two's complement representation of the integer.
     * Does not come with a prefix.
     */
    toReverseTwos(): string;
    /**
     * Returns the arabic numerical representation of the integer as a string.
     */
    toString(): string;
    /**
     * Converts the BigInteger into a decimal number by shifting the decimal place to the left.
     * @param decimals - Number of decimals places
     *
     * @example
     * const bigNumber = BigInteger.fromNumber(100000000);
     * console.log(bigNumber.toDecimal(8)); // 1.00000000
     */
    toDecimal(decimals: number): string;
    /**
     * Adds the other value to this value and returns the result as a new BigInteger.
     * @param other - other operand to add.
     */
    add(other: BigInteger | number): BigInteger;
    /**
     * Subtracts the other value from this value and returns the result as a new BigInteger.
     * @param other - other operand to subtract.
     */
    sub(other: BigInteger | number): BigInteger;
    /**
     * Multiply the other value with this value and returns the result as a new BigInteger.
     * @param other - other operand to multiply.
     */
    mul(other: BigInteger | number): BigInteger;
    /**
     * Divides this value with the other value and returns the result as a new BigInteger.
     * There are no decimals and results are always rounded down.
     * @param other - other operand to divide.
     */
    div(other: BigInteger | number): BigInteger;
    /**
     * Performs the MOD operation with the other value and returns the result as a new BigInteger.
     * @param other - other operand to perform mod.
     */
    mod(other: BigInteger | number): BigInteger;
    /**
     * Compares the two values and returns -1, 0, 1 if this value is larger, equal, smaller than the other value respectively.
     * @param other - other operand to compare against.
     */
    compare(other: BigInteger | number): number;
    equals(other: BigInteger | number): boolean;
}
/**
 * Converts a number to a Fixed8 format hex string
 * @param num - the value to convert
 * @param size - output size in bytes
 * @returns number in Fixed8 representation.
 *
 * @deprecated BigInteger.fromDecimal(num,8).toHex()
 */
export declare function num2fixed8(num: number, size?: number): string;
/**
 * Converts a Fixed8 hex string to its original number
 * @param fixed8hex - number in Fixed8 representation
 *
 * @deprecated  parseInt(BigInteger.fromTwos(fixed8hex, true).toDecimal(8))
 */
export declare function fixed82num(fixed8hex: string): number;
//# sourceMappingURL=BigInteger.d.ts.map