import { HexString, BigInteger } from "../../u";
import { Account } from "../../wallet";
import { TransactionAttribute, TransactionAttributeLike, Witness, WitnessLike, TransactionAttributeJson, WitnessJson, SignerLike, Signer, SignerJson } from "../components";
import { NeonObject } from "../../model";
export interface TransactionLike {
    version: number;
    nonce: number;
    systemFee: BigInteger | string | number;
    networkFee: BigInteger | string | number;
    validUntilBlock: number;
    signers: SignerLike[];
    attributes: TransactionAttributeLike[];
    witnesses: WitnessLike[];
    script: string;
}
export interface TransactionJson {
    hash?: string;
    size: number;
    version: number;
    nonce: number;
    sender: string;
    sysfee: string;
    netfee: string;
    validuntilblock: number;
    signers: SignerJson[];
    attributes: TransactionAttributeJson[];
    script: string;
    witnesses: WitnessJson[];
}
export declare class Transaction implements NeonObject<TransactionLike> {
    #private;
    /**
     * Only version=0 is valid for NEO3
     */
    version: number;
    /**
     * A random 4-byte number to avoid hash collision, range [0, 2**32)
     */
    nonce: number;
    /**
     * transation invoker in script hash in big endian
     */
    get sender(): HexString;
    /**
     * systemFee is calculated by summarizing prices of all the opcodes and interopServices used while executing transaction script in vm.
     *
     * @remarks
     * The most reliable way to calculate minimum systemFee is to use invokeScript method to test, as it's hard to know what the contract will do.
     * If transaction only invokes native contracts, systemFee can be calculated offline. It is distributed to NEO holders.
     *
     * @example
     * const systemFee = SUM(OpCodePrices + InteropServiceCodePrices)
     */
    systemFee: BigInteger;
    /**
     * networkFee is calculated according to transaction size and verificationScript cost in witnesses.
     *
     * @remarks
     * First part of networkFee is counted by transaction size by unit price `FeePerByte`
     * `verificationScriptCost` is calculated by summing up opcodes and interopService prices, like `systemFee`; contract verificationScript may need to be run in the VM to get the exact price.
     * It is distributed to consensus nodes.
     *
     * @example
     * const networkFee = FeePerByte * txSize + SUM(verificationScriptCost)
     *
     */
    networkFee: BigInteger;
    /**
     * Current transaction will be invalid after block of height validUntilBlock
     */
    validUntilBlock: number;
    attributes: TransactionAttribute[];
    signers: Signer[];
    witnesses: Witness[];
    script: HexString;
    /**
     * Maximum duration in blocks that a transaction can stay valid in the mempool.
     * This is 24 hours based on 15s blocktime.
     */
    static MAX_TRANSACTION_LIFESPAN: number;
    static fromJson(input: TransactionJson): Transaction;
    constructor(tx?: Partial<Pick<TransactionLike | Transaction, keyof TransactionLike>>);
    get [Symbol.toStringTag](): string;
    /**
     * Transaction hash.
     */
    hash(): string;
    get size(): number;
    /**
     * Returns the fees as a integer string.
     * Divide this by 100000000 to get the decimal value.
     */
    get fees(): string;
    /**
     * Transaction header size
     */
    get headerSize(): number;
    /**
     * Deserializes a hexstring into a Transaction object.
     * @param hexstring - hexstring of the transaction.
     */
    static deserialize(hex: string): Transaction;
    addSigner(newSigner: SignerLike | Signer): this;
    addAttribute(attr: TransactionAttributeLike): this;
    /**
     * Adds Witness to the Transaction and automatically sorts the witnesses.
     * If the witness already exists, attempts to override the invocationScript
     * with the newly provided one.
     * @param obj - Witness object to add as witness
     */
    addWitness(obj: WitnessLike | Witness): this;
    private orderWitnesses;
    /**
     * Serialize the transaction and return it as a hexstring.
     * @param signed - whether to serialize the signatures. Signing requires it to be serialized without the signatures.
     */
    serialize(signed?: boolean): string;
    /**
     * Signs a transaction.
     * @param signingKey - Account, privateKey or WIF
     * @param networkMagic - Magic number of network found in protocol.json.
     * @param k - optional nonce for signature generation. Setting this causes the signature to be deterministic.
     */
    sign(signingKey: Account | string, networkMagic?: number, k?: string | number): this;
    /**
     * Hexstring of transaction for signing.
     * @param networkMagic - magic number of network found in protocol.json.
     */
    getMessageForSigning(networkMagic: number): string;
    equals(other: Partial<TransactionLike | Transaction>): boolean;
    export(): TransactionLike;
    toJson(): TransactionJson;
    getScriptHashesForVerifying(): string[];
}
export default Transaction;
//# sourceMappingURL=Transaction.d.ts.map