import { HexString, StringStream } from "../u";
import { ContractParam } from "./ContractParam";
import { OpCode } from "./OpCode";
import { InteropServiceCode } from "./InteropServiceCode";
import { ContractCall, ContractCallJson } from "./types";
import { CallFlags } from "./CallFlags";
/**
 * Builds a VM script in hexstring. Used for constructing smart contract method calls.
 */
export declare class ScriptBuilder extends StringStream {
    /**
     * Returns a copy of the script.
     */
    build(): string;
    /**
     * Appends an Opcode, followed by args
     */
    emit(op: OpCode, args?: string): this;
    /**
     * Appends a script that calls an operation on a smart contract.
     * @param scriptHash - ScriptHash of the contract to call.
     * @param operation - The operation to call as a UTF8 string.
     * @param args - Any arguments to pass to the operation.
     * @param callFlags - Flags used while calling the operation.
     *
     * @deprecated Please use emitContractCall which is better typed.
     */
    emitAppCall(scriptHash: string | HexString, operation: string, args?: unknown[], callFlags?: CallFlags): this;
    emitSysCall(service: InteropServiceCode, ...args: unknown[]): this;
    /**
     * Appends data depending on type. Used to append params/array lengths.
     */
    emitPush(data: unknown): this;
    emitBoolean(bool: boolean): this;
    /**
     * Private method to append an array
     */
    private emitArray;
    /**
     * Appends a bytearray.
     */
    emitBytes(byteArray: ArrayBuffer | ArrayLike<number>): this;
    /**
     * Appends a UTF-8 string.
     */
    emitString(str: string): this;
    /**
     * Appends a hexstring.
     *
     * @remarks
     * If a Javascript string is provided, it is emitted as it is.
     * If a HexString is provided, it is emitted as little-endian.
     */
    emitHexString(hexstr: string | HexString): this;
    /**
     * Helper method to emit a public key.
     *
     * @remarks
     * Conventionally, hexstrings are pushed as little endian into the script. However, for public keys, they are pushed as big endian.
     * This helper method reduces the confusion by hiding this edge case.
     * @param publicKey - 66 character string or a HexString.
     *
     * @example
     * const publicKey = "02028a99826edc0c97d18e22b6932373d908d323aa7f92656a77ec26e8861699ef";
     * const result = new ScriptBuilder()
     *  .emitPublicKey(publicKey)
     *  .build();
     *
     * const publicKeyInHexString = HexString.fromHex(publicKey, false);
     * const sameResult = new ScriptBuilder()
     *  .emitPublicKey(publicKeyInHexString)
     *  .build();
     *
     * console.log(result); // 0c2102028a99826edc0c97d18e22b6932373d908d323aa7f92656a77ec26e8861699ef
     *
     *
     */
    emitPublicKey(publicKey: string | HexString): this;
    /**
     * Appends a number. Maximum number possible is ulong(256 bits of data).
     * @param num - for numbers beyond MAX_INT, please pass in a string instead of a javascript number.
     */
    emitNumber(num: number | string): this;
    private roundToBestIntSize;
    /**
     * Private method to append a ContractParam
     */
    emitContractParam(param: ContractParam): this;
    /**
     * Appends a script that calls an operation on a smart contract.
     * @param contractCall - A ContractCall object emitted from a Contract instance.
     */
    emitContractCall(contractCall: ContractCall | ContractCallJson): this;
    appendScript(script: string): this;
}
export default ScriptBuilder;
//# sourceMappingURL=ScriptBuilder.d.ts.map