import { OpCode } from "./OpCode";
import { BigInteger } from "../u";
/**
 * A token from tokenizing a VM script. Consists of a OpCode and optional params that follow it.
 *
 * @remarks
 * Currently, most of the functionality here deals with breaking down the script correctly to identify the parts which are data (not opCodes).
 * An extension of this would be adding an apply function which we can provide a VM context and that should give us a semi working VM.
 *
 * @example
 *
 * const result = OpToken.fromScript("4101020304");
 * console.log(result[0].prettyPrint()); // SYSCALL 01020304
 */
export declare class OpToken {
    code: OpCode;
    params?: string | undefined;
    /**
     * Tokenizes a VM script into its individual instructions.
     * @param script - VM script to tokenize.
     */
    static fromScript(script: string): OpToken[];
    /**
     * Attempts to convert a OpToken that is parsable to an integer.
     * @param opToken - token with code that is a PUSHINT[0-9]+ or PUSH[0-9]+.
     *
     * @throws Error if OpToken contains an invalid code.
     */
    static parseInt(opToken: OpToken): number;
    /**Creates the OpToken for pushing a variable number onto the stack. */
    static forInteger(n: number | string | BigInteger): OpToken;
    constructor(code: OpCode, params?: string | undefined);
    /**
     * Helps to print the token in a formatted way.
     *
     * @remarks
     * Longest OpCode is 12 characters long so default padding is set to 12 characters.
     * This padding does not include an mandatory space between the OpCode and parameters.
     * Padding only happens to instructions with parameters.
     *
     * @example
     * ```
     * const script = "210c0500000000014101020304"
     * console.log(OpToken.fromScript(script).map(t => t.prettyPrint()));
     * //NOP
     * //PUSHDATA1     0000000001
     * //SYSCALL       01020304
     *
     * console.log(OpToken.fromScript(script).map(t => t.prettyPrint(8))); //underpad
     * //NOP
     * //PUSHDATA1 0000000001
     * //SYSCALL  01020304
     * ```
     */
    prettyPrint(padding?: number): string;
    /**
     * Converts an OpToken back to hexstring.
     */
    toScript(): string;
}
//# sourceMappingURL=OpToken.d.ts.map