import { BigInteger, HexString } from "../u";
import { NeonObject } from "../model";
export declare enum ContractParamType {
    Any = 0,
    Boolean = 16,
    Integer = 17,
    ByteArray = 18,
    String = 19,
    Hash160 = 20,
    Hash256 = 21,
    PublicKey = 22,
    Signature = 23,
    Array = 32,
    Map = 34,
    InteropInterface = 48,
    Void = 255
}
export interface ContractParamJson {
    type: string;
    value?: string | boolean | number | ContractParamJson[] | null;
}
export declare type ContractParamLike = Pick<ContractParamJson | ContractParam, keyof ContractParamJson>;
/**
 * These are the parameters used for interacting with smart contracts.
 * Depending on the type, the data is stored differently.
 * The constructor only validates the input types. It does not do transformation.
 * The static methods provide safe parsing from various data types into their intended final storage form.
 *
 * @example
 *
 * ContractParam.integer(1);
 * ContractParam.boolean(true);
 * ContractParam.string("12ab");
 */
export declare class ContractParam implements NeonObject<ContractParamLike> {
    static any(value?: string | HexString | null): ContractParam;
    /**
     * Creates a String ContractParam.
     *
     * @param value - UTF8 string.
     */
    static string(value: string): ContractParam;
    /**
     * Creates a Boolean ContractParam. Does basic checks to convert value into a boolean. Value field will be a boolean.
     */
    static boolean(value: boolean | string | number): ContractParam;
    /**
     * Creates a PublicKey ContractParam. Both encoding formats are allowed. Value field will be a HexString.
     * @param value - A public key (both encoding formats accepted)
     */
    static publicKey(value: string | HexString): ContractParam;
    /**
     * Creates a Hash160 ContractParam. This is used for containing a scriptHash. Value field will be a HexString.
     * Do not reverse the input if using this format.
     * @param value - A 40 character (20 bytes) hexstring. Automatically converts an address to scripthash if provided.
     */
    static hash160(value: string | HexString): ContractParam;
    /**
     * Creates a Hash256 ContractParam. Value field will be a HexString.s
     * @param value - A 64 character (32 bytes) hexstring .
     */
    static hash256(value: string | HexString): ContractParam;
    /**
     * Creates an Integer ContractParam. This is converted into a BigInteger in NeoVM. Value field will be a string.
     * @param value - A value that can be parsed to a BigInteger. Numbers or numeric strings are accepted.
  
     * @example
     * ContractParam.integer(128);
     * ContractParam.integer("128");
     * ContractParam.integer(BigInteger.fromNumber(128));
     */
    static integer(value: string | number | BigInteger): ContractParam;
    /**
     * Creates a ByteArray ContractParam. Value field will be a HexString.
     * @param value - a base64 encoded string (LE) or HexString.
     */
    static byteArray(value: string | HexString): ContractParam;
    /**
     * Creates a Void ContractParam. Value field will be set to null.
     */
    static void(): ContractParam;
    /**
     * Creates an Array ContractParam. Value field will be a ContractParam array.
     * @param params - params to be encapsulated in an array.
     */
    static array(...params: ContractParamLike[]): ContractParam;
    type: ContractParamType;
    value: string | boolean | HexString | ContractParam[] | null;
    constructor(input: ContractParamLike);
    get [Symbol.toStringTag](): string;
    /**
     * Creates a ContractParam from a JSON representation. Use this as the main entry point for conversion from external systems.
     * @param json - JSON format
     */
    static fromJson(json: ContractParamLike): ContractParam;
    export(): ContractParamJson;
    /**
     * Converts the object to JSON format.
     */
    toJson(): ContractParamJson;
    /**
     * Compares whether the other object is equal in value.
     * @param other - ContractParam or the JSON format.
     */
    equals(other: ContractParamLike): boolean;
}
export default ContractParam;
export declare function likeContractParam(cp: Partial<ContractParamLike>): cp is ContractParamLike;
//# sourceMappingURL=ContractParam.d.ts.map