import { Query, } from "../Query";
import { RpcDispatcher } from "./RpcDispatcher";
// eslint-disable-next-line @typescript-eslint/explicit-function-return-type, @typescript-eslint/explicit-module-boundary-types
export function NeoServerRpcMixin(base) {
    return class NeoServerRpcInterface extends base {
        //#region Blockchain
        /**
         * Get the latest block hash.
         */
        async getBestBlockHash() {
            const response = await this.execute(Query.getBestBlockHash());
            return response;
        }
        async getBlock(indexOrHash, verbose) {
            return verbose
                ? await this.execute(Query.getBlock(indexOrHash, 1))
                : await this.execute(Query.getBlock(indexOrHash, 0));
        }
        /**
         * Gets the block hash at a given height.
         */
        async getBlockHash(index) {
            const response = await this.execute(Query.getBlockHash(index));
            return response;
        }
        /**
         * Get the current block height.
         */
        async getBlockCount() {
            const response = await this.execute(Query.getBlockCount());
            return response;
        }
        async getBlockHeader(indexOrHash, verbose) {
            return verbose
                ? await this.execute(Query.getBlockHeader(indexOrHash, 1))
                : await this.execute(Query.getBlockHeader(indexOrHash, 0));
        }
        /**
         * Get the list of public keys in the committee.
         */
        async getCommittee() {
            return await this.execute(Query.getCommittee());
        }
        /**
         * Gets the state of the contract at the given scriptHash.
         */
        async getContractState(scriptHash) {
            const response = await this.execute(Query.getContractState(scriptHash));
            return response;
        }
        async getRawMemPool(shouldGetUnverified = 0) {
            return shouldGetUnverified
                ? await this.execute(Query.getRawMemPool(1))
                : await this.execute(Query.getRawMemPool(0));
        }
        async getRawTransaction(txid, verbose) {
            return verbose
                ? await this.execute(Query.getRawTransaction(txid, 1))
                : await this.execute(Query.getRawTransaction(txid, 0));
        }
        /**
         * Gets the corresponding value of a key in the storage of a contract address.
         */
        async getStorage(scriptHash, key) {
            const response = await this.execute(Query.getStorage(scriptHash, key));
            return response;
        }
        /**
         * Gets the block index in which the transaction is found.
         * @param txid - hash of the specific transaction.
         */
        async getTransactionHeight(txid) {
            const response = await this.execute(Query.getTransactionHeight(txid));
            return response;
        }
        //#endregion
        //#region Node
        /**
         * Gets the number of peers this node is connected to.
         */
        async getConnectionCount() {
            const response = await this.execute(Query.getConnectionCount());
            return response;
        }
        /**
         * Gets the list of validators available for voting.
         */
        async getNextBlockValidators() {
            const response = await this.execute(Query.getNextBlockValidators());
            return response;
        }
        /**
         * Gets a list of all peers that this node has discovered.
         */
        async getPeers() {
            const response = await this.execute(Query.getPeers());
            return response;
        }
        /**
         * Gets the version of the NEO node along with various other metadata.
         */
        async getVersion() {
            const response = await this.execute(Query.getVersion());
            return response;
        }
        /**
         * Sends a serialized transaction to the network.
         * @returns transaction id
         */
        async sendRawTransaction(transaction) {
            const response = await this.execute(Query.sendRawTransaction(transaction));
            return response.hash;
        }
        /**
         * Submits a serialized block to the network.
         * @returns block hash if success
         */
        async submitBlock(block) {
            const response = await this.execute(Query.submitBlock(block));
            return response.hash;
        }
        //#endregion
        //#region SmartContract
        /**
         * Get the amount of unclaimed GAS for a NEO address. This is returned as the raw value. To get the display value, divide this by 100000000.
         */
        async getUnclaimedGas(addr) {
            const response = await this.execute(Query.getUnclaimedGas(addr));
            return response.unclaimed;
        }
        /**
         * Submits a verification script to run under a contract.
         * @param scriptHash - contract to test
         * @param args - arguments to pass
         * @param signers - Signers to be included in transaction
         */
        async invokeContractVerify(scriptHash, args, signers = []) {
            return await this.execute(Query.invokeContractVerify(scriptHash, args, signers));
        }
        /**
         * Submits a contract method call with parameters for the node to run. This method is a local invoke, results are not reflected on the blockchain.
         */
        async invokeFunction(scriptHash, operation, params = [], signers = []) {
            return await this.execute(Query.invokeFunction(scriptHash, operation, params, signers));
        }
        /**
         * Submits a script for the node to run. This method is a local invoke, results are not reflected on the blockchain.
         *
         * @param script - base64-encoded hexstring.
         * @param signers - signatures accompanying this execution.
         */
        async invokeScript(script, signers = []) {
            return await this.execute(Query.invokeScript(script, signers));
        }
        //#endregion
        //#region Utilities
        /**
         * Returns a list of plugins loaded by the node.
         */
        async listPlugins() {
            const response = await this.execute(Query.listPlugins());
            return response;
        }
        /**
         * Checks if the provided address is a valid NEO address.
         */
        async validateAddress(addr) {
            const response = await this.execute(Query.validateAddress(addr));
            return response.isvalid;
        }
    };
}
/**
 * RPC Client model to query a NEO node. Contains built-in methods to query using RPC calls.
 */
export class NeoServerRpcClient extends NeoServerRpcMixin(RpcDispatcher) {
    get [Symbol.toStringTag]() {
        return `NeoServerRpcClient(${this.url})`;
    }
}
//# sourceMappingURL=NeoServerRpcClient.js.map