import { Transaction, TransactionJson, Signer, SignerJson } from "../tx";
import { ContractManifestJson, StackItemJson, NEFJson, ContractParamJson } from "../sc";
import { BlockJson, Validator, BlockHeaderJson } from "../types";
import { HexString } from "../u";
export declare type BooleanLikeParam = 0 | 1 | boolean;
export interface QueryLike<T extends unknown[]> {
    method: string;
    params: T;
    id: number;
    jsonrpc: "2.0";
}
export interface RPCResponse<T> {
    jsonrpc: string;
    id: number;
    result: T;
    error?: RPCErrorResponse;
}
export interface RPCErrorResponse {
    code: number;
    message: string;
}
export interface ApplicationLogJson {
    txid: string;
    executions: {
        trigger: string;
        vmstate: string;
        gasconsumed: string;
        stack?: StackItemJson[];
        notifications: {
            contract: string;
            eventname: string;
            state: ContractParamJson;
        }[];
    }[];
}
/**
 * Result from calling invokescript or invokefunction.
 */
export interface InvokeResult {
    /** The script that is sent for execution on the blockchain as a base64 string. */
    script: string;
    /** State of VM on exit. HALT means a successful exit while FAULT means exit with error. */
    state: "HALT" | "FAULT";
    /** Amount of gas consumed up to the point of stopping in the VM. If state is FAULT, this value is not representative of the amount of gas it will consume if it somehow succeeds on the blockchain.
     * This is a decimal value.
     */
    gasconsumed: string;
    /** A human-readable string clarifying the exception that occurred. Only available when state is "FAULT". */
    exception: string | null;
    stack: StackItemJson[];
    /** A ready to send transaction that wraps the script.
     * Only available when signers are provided and the sender's private key is open in the RPC node.
     * Formatted in base64-encoding.
     */
    tx?: string;
}
export interface GetContractStateResult {
    id: number;
    updatecounter: number;
    /** 0x prefixed hexstring */
    hash: string;
    /** Base64 encoded string */
    nef: NEFJson;
    manifest: ContractManifestJson;
}
export interface GetNep17TransfersResult {
    sent: Nep17TransferEvent[];
    received: Nep17TransferEvent[];
    address: string;
}
export interface Nep17TransferEvent {
    timestamp: number;
    assethash: string;
    transferaddress: string;
    amount: string;
    blockindex: number;
    transfernotifyindex: number;
    txhash: string;
}
export interface GetNep17BalancesResult {
    address: string;
    balance: {
        assethash: string;
        amount: string;
        lastupdatedblock: number;
    }[];
}
export interface GetPeersResult {
    unconnected: NodePeer[];
    bad: NodePeer[];
    connected: NodePeer[];
}
export interface GetRawMemPoolResult {
    height: number;
    verified: string[];
    unverified: string[];
}
export interface GetRawTransactionResult extends TransactionJson {
    hash: string;
    blockhash: string;
    /** Number of blocks that has been confirmed between blocktime and now. */
    confirmations: number;
    /** Unix timestamp in milliseconds. */
    blocktime: number;
    vm_state: "HALT" | "FAULT";
}
export interface GetVersionResult {
    tcpport: number;
    wsport: number;
    nonce: number;
    useragent: string;
    protocol: {
        addressversion: number;
        network: number;
        validatorscount: number;
        msperblock: number;
        maxtraceableblocks: number;
        maxvaliduntilblockincrement: number;
        maxtransactionsperblock: number;
        memorypoolmaxtransactions: number;
        initialgasdistribution: number;
    };
}
export interface NodePeer {
    /** IPv4 Address */
    address: string;
    port: number;
}
export interface CliPlugin {
    name: string;
    version: string;
    interfaces: string[];
}
export interface SendResult {
    /** 0x prefixed hexstring */
    hash: string;
}
export interface ValidateAddressResult {
    address: string;
    isvalid: boolean;
}
export interface GetUnclaimedGasResult {
    unclaimed: string;
    address: string;
}
/**
 * A Query object helps us to construct and record requests for the Neo node RPC. For each RPC endpoint, the equivalent static method is camelcased. Each Query object can only be used once.
 *
 * @example
 * const q1 = Query.getBestBlockHash();
 */
export declare class Query<TParams extends unknown[], TResponse> {
    /**
     * This Query returns the hash of the highest block.
     */
    static getBestBlockHash(): Query<[], string>;
    /**
     * Query returning the application log.
     */
    static getApplicationLog(hash: string): Query<[string], ApplicationLogJson>;
    /**
     * This Query returns the specified block either as a hexstring or human readable JSON.
     * @param indexOrHash - height or hash of block.
     * @param verbose - 0 for hexstring, 1 for JSON. Defaults to 0.
     */
    static getBlock(indexOrHash: number | string, verbose: 1 | true): Query<[number | string, 1 | true], BlockJson>;
    static getBlock(indexOrHash: number | string, verbose?: 0 | false): Query<[number | string, 0 | false], string>;
    /**
     * This Query returns the current block height.
     */
    static getBlockCount(): Query<[], number>;
    /**
     * This Query returns the hash of a specific block.
     * @param index - height of block.
     */
    static getBlockHash(index: number): Query<[number], string>;
    /**
     * This Query Returns the corresponding block header information according to the specified script hash.
     * @param indexOrHash - height or hash of block.
     * @param verbose - Optional, the default value of verbose is 0. When verbose is 0, the serialized information of the block is returned, represented by a hexadecimal string. If you need to get detailed information, you will need to use the SDK for deserialization. When verbose is 1, detailed information of the corresponding block in Json format string, is returned.
     */
    static getBlockHeader(indexOrHash: string | number, verbose: 1 | true): Query<[string | number, 1 | true], BlockHeaderJson>;
    static getBlockHeader(indexOrHash: string | number, verbose?: 0 | false): Query<[string | number, 0 | false], string>;
    /**
     * This Query returns the list of public keys in the current committee.
     */
    static getCommittee(): Query<[], string[]>;
    /**
     * This Query returns the number of other nodes that this node is connected to.
     */
    static getConnectionCount(): Query<[], number>;
    /**
     * This Query returns information about the smart contract registered at the specific hash.
     * @param scriptHash - hash of contract
     */
    static getContractState(scriptHash: string): Query<[string], GetContractStateResult>;
    /**
     * Query returning the Nep17 transfer history of an account. Defaults of 1 week of history.
     * @param accountIdentifer - address or scriptHash of account
     * @param startTime - millisecond Unix timestamp
     * @param endTime - millisecond Unix timestamp
     */
    static getNep17Transfers(accountIdentifer: string, startTime?: string, endTime?: string): Query<[string, string?, string?], GetNep17TransfersResult>;
    /**
     * Query returning the Nep17 balance of an account.
     * @param accountIdentifer - address or scriptHash of account
     */
    static getNep17Balances(accountIdentifer: string): Query<[string], GetNep17BalancesResult>;
    /**
     * This Query returns the list of nodes that this node is connected to.
     */
    static getPeers(): Query<[], GetPeersResult>;
    /**
     * This Query returns the transaction hashes of the transactions confirmed or unconfirmed.
     * @param shouldGetUnverified - Optional. Default is 0.
     *
     * shouldGetUnverified = 0, get confirmed transaction hashes
     *
     * shouldGetUnverified = 1, get current block height and confirmed and unconfirmed tx hash
     */
    static getRawMemPool(shouldGetUnverified?: BooleanLikeParam): Query<[0 | false], string[]>;
    static getRawMemPool(shouldGetUnverified: 1 | true): Query<[1], GetRawMemPoolResult>;
    /**
     * This Query returns information about a specific transaction in either hexstring or human readable JSON.
     * @param txid - hash of the specific transaction.
     * @param verbose - 0 for hexstring, 1 for JSON. Defaults to 0.
     */
    static getRawTransaction(txid: string, verbose?: 0 | false): Query<[string, 0 | false], string>;
    static getRawTransaction(txid: string, verbose: 1 | true): Query<[string, 1 | true], GetRawTransactionResult>;
    /**
     * This Query returns the value stored at the specific key under a specific contract in base64 format.
     * @param scriptHash - hash of contract.
     * @param key - the storage key in as hex string
     */
    static getStorage(scriptHash: string, key: string): Query<[string, string], string>;
    /**
     * This Query returns the block index in which the transaction is found.
     * @param txid - hash of the specific transaction.
     */
    static getTransactionHeight(txid: string): Query<[string], number>;
    /**
     * Gets the list of candidates available for voting.
     * @returns list of validators
     */
    static getNextBlockValidators(): Query<[], Validator[]>;
    /**
     * Returns the node version.
     */
    static getVersion(): Query<[], GetVersionResult>;
    /**
     * Invoke the verification portion of a contract.
     * @param scriptHash - hash of contract to test
     * @param args - arguments to pass
     * @param signers - Signers to be included in transaction
     */
    static invokeContractVerify(scriptHash: string, args?: unknown[], signers?: (Signer | SignerJson)[]): Query<[string, unknown[], SignerJson[]], InvokeResult>;
    /**
     * This Query invokes the VM to run the specific contract with the provided operation and params. Do note that this function only suits contracts with a Main(string, args[]) entry method.
     * @param scriptHash - hash of contract to test.
     * @param operation - name of operation to call (first argument)
     * @param params - parameters to pass (second argument). ContractParam objects will be exported to JSON format.
     * @param signers - scripthashes of witnesses that should sign the transaction containing this script. Signers will be exported to JSON format.
     */
    static invokeFunction(scriptHash: string, operation: string, params?: unknown[], signers?: (Signer | SignerJson)[]): Query<[string, string, unknown[], SignerJson[]], InvokeResult>;
    /**
     * This Query runs the specific script through the VM.
     * @param script - base64-encoded string.
     * @param signers - scripthashes of witnesses that should sign the transaction containing this script.
     */
    static invokeScript(script: string | HexString, signers?: (Signer | SignerJson)[]): Query<[string, SignerJson[]], InvokeResult>;
    /**
     * This Query returns a list of plugins loaded by the node.
     */
    static listPlugins(): Query<[], CliPlugin[]>;
    /**
     * This Query transmits the specific transaction to the node. On success, the node will return the transaction hash.
     * @param transaction - transaction as a Transaction object or base64 hexstring.
     */
    static sendRawTransaction(transaction: Transaction | string | HexString): Query<[string], SendResult>;
    /**
     * This Query submits a block for processing.
     * @param block - a serialized block
     */
    static submitBlock(block: string): Query<[string], SendResult>;
    /**
     * This Query submits an address for validation.
     * @param addr - address to validate.
     */
    static validateAddress(addr: string): Query<[string], ValidateAddressResult>;
    /**
     * This Query returns the available unclaimed bonus GAS for a NEO address
     * @param addr - a NEO address
     */
    static getUnclaimedGas(addr: string): Query<[string], GetUnclaimedGasResult>;
    id: number;
    method: string;
    params: TParams;
    constructor(req: Partial<QueryLike<TParams>>);
    get [Symbol.toStringTag](): string;
    export(): QueryLike<TParams>;
    equals(other: Partial<QueryLike<TParams>>): boolean;
}
export default Query;
//# sourceMappingURL=Query.d.ts.map